﻿using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using IDI.Demo.OrderPlacementServiceReference1;

namespace IDI.Demo
{
    public class OPClient
    {
        #region Constructor

        public OPClient(string environment, long serviceNumberBase)
        {
            this.environmentName = environment;
            this.serviceNumberBase = serviceNumberBase;

            securityToken = Utilities.CreateSecurityToken(environmentName);
            channel = Utilities.CreateOrderPlacementChannel(securityToken);
            requestor = new Requestor
            {
                AuthorizationToken = "AuthorizationToken"
            };
        }

        #endregion

        // Hard-coded values used in this example
        private string environmentName = null;

        // This value needs to be incremented each test run to ensure unique service numbers.  We use a variable because there are multiple services
        // being added in this example and it is annoying to have to update them all every time.
        private long serviceNumberBase = 0;

        // Values we create common to all requests
        SecurityToken securityToken = null;
        OrderPlacementServicePortTypeChannel channel = null;
        Requestor requestor = null;

        #region Public Methods

        #region CreateCart

        public int CreateCart()
        {
            // Block 1: Create a new cart
            CreateCartReq createCartReq = new CreateCartReq()
            {
                CustomerID = new AuthenticatedID
                {
                    AuthenticatedKey = "24461",
                    ID = 24461
                },
                AdditionalCartInformation = new AdditionalCartInformation
                {
                    Description = "Cart for Order Placement sample project",
                    OrderTypeID = "1038"
                }
            };
            CreateCartRequest createCartRequest = new CreateCartRequest
            {
                Request = createCartReq,
                Requestor = requestor
            };
            CreateCartResponse createCartResponse = channel.CreateCart(createCartRequest);
            CheckForErrorsInResponse(createCartResponse.Messages);
            CreateCartResult createCartResult = createCartResponse.Result;

            // Save the cart ID for later use.  We're going to be using it on every subsequent request.
            int cartID = createCartResult.Cart.CartID.ID;
            return cartID;
        }

        #endregion

        #region Add Service (no features)

        public int AddService(int cartID) { 
            // Block 2: Add a new service to the cart
            ModifyCartReq modifyCartReq = new ModifyCartReq()
            {
                AdditionalCartInformation = new AdditionalCartInformation
                {
                    ServiceInformation = new ServiceInformation()
                },
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                },
                ResponseGroups = new CartResponseGroups()
            };
            modifyCartReq.AdditionalCartInformation.ServiceInformation.Add(new ServiceInformationItem()
            {
                CatalogID = 24392,
                Description = "VOIP Line",
                ServiceContact = new Contact
                {
                    ContactID = "1145722"
                },
                ServiceNumber = serviceNumberBase.ToString(),   // Comment out this line to see what an error looks like
                StartDate = "2017-09-19T00:00:00.000000"        // Must match the ISO 8601 format
            });
            modifyCartReq.ResponseGroups.Add(CartResponseGroup.CartSummary);
            ModifyCartRequest modifyCartRequest = new ModifyCartRequest
            {
                Request = modifyCartReq,
                Requestor = requestor
            };
            ModifyCartResponse modifyCartResponse = channel.ModifyCart(modifyCartRequest);
            CheckForErrorsInResponse(modifyCartResponse.Messages);

            // Because we used the CartSummary response group the modifyCartResponse.Result.CartSummary data is returned in the response
            ModifyCartResult modifyCartResult = modifyCartResponse.Result;

            // We need to retrieve the ID of the service information item we added for the next request
            int serviceInformationItemID = modifyCartResult.Cart.CartDetails.ServiceDetails[0].ServiceInformationItemID;
            return serviceInformationItemID;
        }

        #endregion

        #region Add Features to Service in Cart

        public void AddFeaturesToService(int cartID, int serviceInformationItemID)
        {
            // Block 3: Add Features to the Cart
            ModifyCartReq modifyCartReq = new ModifyCartReq()
            {
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                },
                PurchaseItems = new PurchaseItems()
            };
            modifyCartReq.PurchaseItems.Add(new PurchaseItem
            {
                AdditionalPurchaseItemInformation = new AdditionalPurchaseItemInformation
                {
                    PriceOverride = 59.99m,
                    ServiceInformationItemID = serviceInformationItemID  // This is how we tie the product to the service
                },
                CatalogID = 24829,
                Quantity = 1
            });
            modifyCartReq.PurchaseItems.Add(new PurchaseItem
            {
                CatalogID = 24322,
                Quantity = 1
            });
            ModifyCartRequest modifyCartRequest = new ModifyCartRequest
            {
                Request = modifyCartReq,
                Requestor = requestor
            };
            ModifyCartResponse modifyCartResponse = channel.ModifyCart(modifyCartRequest);
            CheckForErrorsInResponse(modifyCartResponse.Messages);
        }

        #endregion

        #region Add Service with Features

        public int AddServiceWithFeatures(int cartID)
        {
            // Block 4: Add a service with features in the same request
            int temporaryServiceInformationItemID = -1;
            ModifyCartReq modifyCartReq = new ModifyCartReq()
            {
                AdditionalCartInformation = new AdditionalCartInformation
                {
                    ServiceInformation = new ServiceInformation()
                },
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                },
                PurchaseItems = new PurchaseItems(),
                ResponseGroups = new CartResponseGroups()
            };
            modifyCartReq.AdditionalCartInformation.ServiceInformation.Add(new ServiceInformationItem()
            {
                CatalogID = 24392,
                Description = "VOIP Line",
                ReplicateQuantity = 3,
                ServiceContact = new Contact
                {
                    ContactID = "1145722"
                },
                ServiceNumber = "XXXXXXXXXX",                       
                StartDate = "2017-09-19T00:00:00.000000",                               // Must match the ISO 8601 format
                TemporaryServiceInformationItemID = temporaryServiceInformationItemID   // This is the temporary ID used in this request to link related objects to this new service
            });
            modifyCartReq.PurchaseItems.Add(new PurchaseItem
            {
                AdditionalPurchaseItemInformation = new AdditionalPurchaseItemInformation
                {
                    PriceOverride = 59.99m,
                    ServiceInformationItemID = temporaryServiceInformationItemID  // This is how we tie the product to the service
                },
                CatalogID = 24829,
                Quantity = 1
            });
            modifyCartReq.ResponseGroups.Add(CartResponseGroup.CartSummary);
            ModifyCartRequest modifyCartRequest = new ModifyCartRequest
            {
                Request = modifyCartReq,
                Requestor = requestor
            };
            ModifyCartResponse modifyCartResponse = channel.ModifyCart(modifyCartRequest);
            CheckForErrorsInResponse(modifyCartResponse.Messages);
            ModifyCartResult modifyCartResult = modifyCartResponse.Result;

            // We need to retrieve the ID of the service information item we added for the next request
            int serviceInformationItemID = modifyCartResult.Cart.CartDetails.ServiceDetails[0].ServiceInformationItemID;
            return serviceInformationItemID;
        }

        #endregion

        #region Replicate Service

        public void ReplicateService(int cartID, int serviceInformationItemID)
        {
            // Block 4: Replicate a service to 3 services
            ModifyCartReq modifyCartReq = new ModifyCartReq()
            {
                AdditionalCartInformation = new AdditionalCartInformation
                {
                    ReplicateServices = new ReplicateServices()
                },
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                }
            };

            // Replicate our template service with feature three times
            modifyCartReq.AdditionalCartInformation.ReplicateServices.Add(new ReplicateService
            {
                ServiceInformationItemID = serviceInformationItemID,
                ServiceNumber = (++serviceNumberBase).ToString()
            });
            modifyCartReq.AdditionalCartInformation.ReplicateServices.Add(new ReplicateService
            {
                ServiceInformationItemID = serviceInformationItemID,
                ServiceNumber = (++serviceNumberBase).ToString()
            });
            modifyCartReq.AdditionalCartInformation.ReplicateServices.Add(new ReplicateService
            {
                ServiceInformationItemID = serviceInformationItemID,
                ServiceNumber = (++serviceNumberBase).ToString()
            });
            ModifyCartRequest modifyCartRequest = new ModifyCartRequest
            {
                Request = modifyCartReq,
                Requestor = requestor
            };

            ModifyCartResponse modifyCartResponse = channel.ModifyCart(modifyCartRequest);
            CheckForErrorsInResponse(modifyCartResponse.Messages);
            ModifyCartResult modifyCartResult = modifyCartResponse.Result;
        }

        #endregion

        #region Remove Service from Cart

        public void RemoveService(int cartID, int serviceInformationItemID)
        {
            // Block 4: Replicate a service to 3 services
            ModifyCartReq modifyCartReq = new ModifyCartReq()
            {
                AdditionalCartInformation = new AdditionalCartInformation
                {
                    ServiceInformation = new ServiceInformation()
                },
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                }
            };
            modifyCartReq.AdditionalCartInformation.ServiceInformation.Add(new ServiceInformationItem
            {
                Remove = true,
                ServiceInformationItemID = serviceInformationItemID
            });
            
            ModifyCartRequest modifyCartRequest = new ModifyCartRequest
            {
                Request = modifyCartReq,
                Requestor = requestor
            };

            ModifyCartResponse modifyCartResponse = channel.ModifyCart(modifyCartRequest);
            CheckForErrorsInResponse(modifyCartResponse.Messages);
            ModifyCartResult modifyCartResult = modifyCartResponse.Result;
        }

        #endregion

        #region Checkout

        public void Checkout(int cartID)
        {
            CheckoutReq checkoutReq = new CheckoutReq
            {
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                }
            };

            CheckoutRequest checkoutRequest = new CheckoutRequest
            {
                Request = checkoutReq,
                Requestor = requestor
            };

            CheckoutResponse checkoutResponse = channel.Checkout(checkoutRequest);
            CheckForErrorsInResponse(checkoutResponse.Messages);
        }

        #endregion

        #region SubmitCart

        public void SubmitCart(int cartID)
        {
            SubmitCartReq submitCartReq = new SubmitCartReq
            {
                CartID = new AuthenticatedID
                {
                    AuthenticatedKey = cartID.ToString(),
                    ID = cartID
                }
            };

            SubmitCartRequest submitCartRequest = new SubmitCartRequest
            {
                Request = submitCartReq,
                Requestor = requestor
            };

            SubmitCartResponse submitCartResponse = channel.SubmitCart(submitCartRequest);
            CheckForErrorsInResponse(submitCartResponse.Messages);
        }

        #endregion

        #endregion

        #region Private Methods

        #region CheckForErrorsInResponse

        private void CheckForErrorsInResponse(Messages messages)
        {
            if (messages != null)
            {
                foreach (Message message in messages)
                {
                    if (message.Type == MessageType.Error)
                    {
                        throw new Exception(string.Format("Error {0}: {1}", message.LoggingNumber, message.MessageMember));
                    }
                }
            }
        }

        #endregion

        #endregion
    }
}
